// Lotta Auto Welcome (Content Script)
// Heuristic DOM strategy to work across small UI changes.
// - Sends a welcome message on each new pairing
// - Waits N seconds for an incoming reply; if none, clicks Next/Skip
// - N and message are configurable via chrome.storage

(function () {
  const DEFAULTS = {
    welcomeMessage: "Hey! 😊 How's it going?",
    waitSeconds: 20,
    debug: false
  };

  let state = {
    pairedId: null,        // increments on each pairing we detect
    haveSentWelcome: false,
    replyTimer: null,
    waitingForReply: false,
    settings: { ...DEFAULTS }
  };

  const log = (...args) => {
    if (state.settings.debug) console.log("[LottaAutoWelcome]", ...args);
  };

  // --- Settings ---
  chrome.storage.sync.get(DEFAULTS, (cfg) => {
    state.settings = { ...DEFAULTS, ...cfg };
    log("Loaded settings:", state.settings);
    start();
  });

  function start() {
    // Observe DOM for pairing lifecycle and incoming messages
    const obs = new MutationObserver(onMutations);
    obs.observe(document.documentElement || document.body, {
      childList: true,
      subtree: true,
      attributes: true,
      characterData: false
    });

    // Also attempt immediate init in case chat is already ready
    checkPairingAndMaybeSend();
  }

  function onMutations(mutations) {
    // Check for a (re-)pairing event or new message arrival
    let messageArrived = false;

    for (const m of mutations) {
      if (m.addedNodes && m.addedNodes.length) {
        // If a new message bubble appeared that isn’t ours, mark as reply
        for (const n of m.addedNodes) {
          if (isIncomingMessageNode(n)) {
            messageArrived = true;
          }
        }
      }
    }

    if (messageArrived) onIncomingMessage();

    // Also periodically re-check pairing readiness
    checkPairingAndMaybeSend();
  }

  // --- Pairing detection ---
  function checkPairingAndMaybeSend() {
    const chatReady = isChatReady();
    if (!chatReady) return;

    // Create a session key from DOM we can detect (e.g., partner indicator) or fallback increment
    const partnerKey = getPartnerKey();

    if (partnerKey && partnerKey !== state.pairedId) {
      log("New pairing detected:", partnerKey);
      resetSession(partnerKey);
      sendWelcome();
      startReplyTimer();
    }
  }

  function resetSession(partnerKey) {
    state.pairedId = partnerKey;
    state.haveSentWelcome = false;
    clearReplyTimer();
    state.waitingForReply = false;
  }

  function isChatReady() {
    // Heuristic: input exists and is enabled
    const input = findInput();
    const nextBtn = findNextButton();
    const connected = !!document.querySelector('[data-status="connected"], .connected, .chat-connected');
    return (!!input && !input.disabled && !!nextBtn) || connected;
  }

  function getPartnerKey() {
    // Try to find an element that changes per partner (username/ID/banner)
    const partnerEl = document.querySelector('[data-partner-id], .partner-id, .chat-header, .status, .room-id');
    if (partnerEl && partnerEl.textContent) return partnerEl.textContent.trim();

    // Fallback: count the number of ‘Connected’ flashes or chat clears
    // Use the current time second-resolution to avoid accidental duplicates
    return `pair-${Date.now()}`;
  }

  // --- Messaging ---
  async function sendWelcome() {
    if (state.haveSentWelcome) return;

    const input = findInput();
    if (!input) {
      log("No input found; will retry on next mutation.");
      return;
    }

    // Focus and set value
    input.focus();
    setNativeValue(input, state.settings.welcomeMessage);

    // Try send via Enter first (works on many chat UIs)
    if (!sendViaEnter(input)) {
      // Fallback to clicking a send button if present
      const sendBtn = findSendButton();
      if (sendBtn) sendBtn.click();
      else {
        // As last resort, try dispatching 'submit' on a wrapping form
        const form = input.closest('form');
        if (form) form.dispatchEvent(new Event('submit', { bubbles: true, cancelable: true }));
      }
    }

    state.haveSentWelcome = true;
    state.waitingForReply = true;
    log("Welcome message sent.");
  }

  function sendViaEnter(el) {
    try {
      const e1 = new KeyboardEvent('keydown', { key: 'Enter', code: 'Enter', bubbles: true });
      const e2 = new KeyboardEvent('keypress', { key: 'Enter', code: 'Enter', bubbles: true });
      const e3 = new KeyboardEvent('keyup', { key: 'Enter', code: 'Enter', bubbles: true });
      el.dispatchEvent(e1); el.dispatchEvent(e2); el.dispatchEvent(e3);
      return true; // if UI is enter-to-send, this will usually work
    } catch (e) {
      return false;
    }
  }

  function setNativeValue(el, value) {
    const last = el.value;
    el.value = value;
    const event = new Event('input', { bubbles: true });
    // Some frameworks require both input and change
    el.dispatchEvent(event);
    if (last !== value) el.dispatchEvent(new Event('change', { bubbles: true }));
  }

  function findInput() {
    // Prefer role textbox or textarea
    let input = document.querySelector('[role="textbox"], textarea');
    if (!input) input = document.querySelector('input[type="text"], input:not([type])');
    return input;
  }

  function findSendButton() {
    // Common button patterns
    const candidates = Array.from(document.querySelectorAll('button, [role="button"]'));
    return candidates.find(b => /send|\u23ce|\u27a4|\u27a1|paper\s*plane/i.test(b.textContent || b.getAttribute('aria-label') || '')) || null;
  }

  function findNextButton() {
    const btns = Array.from(document.querySelectorAll('button, [role="button"]'));
    const match = btns.find(b => /(next|skip|new\s*chat|leave)/i.test((b.textContent || b.getAttribute('aria-label') || '').trim()));
    return match || null;
  }

  function isIncomingMessageNode(n) {
    if (!(n instanceof Element)) return false;

    // Check typical chat bubble classes
    const txt = (n.textContent || '').trim();
    if (!txt) return false;

    const classStr = n.className ? String(n.className) : '';
    if (/(message|msg|bubble|balloon|chat|line)/i.test(classStr)) {
      if (!/(self|me|own|outgoing|mine)/i.test(classStr)) return true; // likely incoming
    }

    // Also check if it looks like a message row with an avatar that isn't ours
    if (n.querySelector && (n.querySelector('.avatar, [data-user]') || n.querySelector('time'))) {
      if (!n.closest('.self, .me, .outgoing')) return true;
    }

    return false;
  }

  function onIncomingMessage() {
    if (!state.waitingForReply) return;
    log("Incoming message detected — cancelling skip.");
    clearReplyTimer();
    state.waitingForReply = false;
  }

  function startReplyTimer() {
    clearReplyTimer();
    const ms = Math.max(3, Number(state.settings.waitSeconds)) * 1000;
    state.replyTimer = setTimeout(() => {
      if (state.waitingForReply) {
        log(`No reply after ${state.settings.waitSeconds}s — moving to next.`);
        skipToNext();
      }
    }, ms);
    log(`Started reply timer for ${state.settings.waitSeconds}s.`);
  }

  function clearReplyTimer() {
    if (state.replyTimer) {
      clearTimeout(state.replyTimer);
      state.replyTimer = null;
    }
  }

  function skipToNext() {
    const btn = findNextButton();
    if (btn) {
      btn.click();
      // After skipping, reset so the next pairing triggers again
      resetSession(null);
      // allow DOM to settle then detect next pair
      setTimeout(checkPairingAndMaybeSend, 1000);
    } else {
      log("Next/Skip button not found.");
    }
  }
})();